﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace YoseenSDKCS
{
    /// <summary>
    /// 温度转位图算法配置标志
    /// </summary>
    [Flags]
    public enum StrechControlFlags : int
    {
        /// <summary>
        /// 算法类型
        /// </summary>
        SCF_StrechType = 0x0001,

        //SCF_Contrast    =    0x0002,
        //SCF_Brightness  =    0x0004,

        /// <summary>
        /// 增益
        /// </summary>
        SCF_Gain = 0x0008,

        //SCF_Linear = 0x0010,

        /// <summary>
        /// PHE流控
        /// </summary>
        SCF_PHEStream = 0x0020,

        /// <summary>
        /// DDE等级
        /// </summary>
        SCF_DDELevel = 0x0040,

        /// <summary>
        /// 温度标记
        /// </summary>
        SCF_ColorTemp = 0x0080,

        /// <summary>
        /// 手动温度
        /// </summary>
        SCF_ManEnable = 0x0100,

        /// <summary>
        /// 手动温度0
        /// </summary>
        SCF_ManTemp0 = 0x0200,

        /// <summary>
        /// 手动温度1
        /// </summary>
        SCF_ManTemp1 = 0x0400,

        SCF_All = 0x0000ffff,
    }

    /// <summary>
    /// 温度转位图算法类型
    /// </summary>
    public enum strech_type : byte
    {
        /// <summary>
        /// PHE
        /// </summary>
        STRECH_TYPE_PHE = 1,

        /// <summary>
        /// LINEAR
        /// </summary>
        STRECH_TYPE_LINEAR,
    }

    /// <summary>
    /// 温度标记类型
    /// </summary>
    public enum ColorTempType : byte
    {
        /// <summary>
        /// 无
        /// </summary>
        None,

        /// <summary>
        /// 高温
        /// </summary>
        High,

        /// <summary>
        /// 低温
        /// </summary>
        Low,

        /// <summary>
        /// 区间内
        /// </summary>
        Mid,

        /// <summary>
        /// 区间外
        /// </summary>
        ExMid,
    }

    /// <summary>
    /// 温度转位图算法配置
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 64)]
    public struct strech_control
    {
        /// <summary>
        /// 算法配置标志
        /// </summary>
        public StrechControlFlags change_type;

        /// <summary>
        /// 算法类型
        /// </summary>
        public strech_type strech_type;

        /// <summary>
        /// 对比度0-100
        /// </summary>
        public byte contrast;

        /// <summary>
        /// 亮度0-100
        /// </summary>
        public byte brightness;

        /// <summary>
        /// PHE流控
        /// </summary>
        public byte phe_stream;

        /// <summary>
        /// 增益0.1-100
        /// </summary>
        public float gain;

        /// <summary>
        /// 手动温度0, 单位0.1C
        /// </summary>
        public short man_temp0;

        /// <summary>
        /// 手动温度1, 单位0.1C
        /// </summary>
        public short man_temp1;

        /// <summary>
        /// 手动温度
        /// </summary>
        public byte man_enable;
        byte pad;

        /// <summary>
        /// DDE等级, [0,8], 0关闭DDE
        /// </summary>
        public byte dde_level;

        /// <summary>
        /// 温度标记类型
        /// 
        /// 高温:	温度高于ct_temp0, 用ct_color0
        /// 低温：	温度低于ct_temp0, 用ct_color0
        /// 区间内： 温度介于[ct_temp0, ct_temp1], 用ct_color0
        /// 区间外:	温度低于ct_temp0, 用ct_color0; 温度高于ct_temp1,用ct_color1
        /// </summary>
        public ColorTempType ct_type;

        /// <summary>
        /// 标记温度0, 单位0.1C
        /// </summary>
        public short ct_temp0;

        /// <summary>
        /// 标记温度1, 单位0.1C
        /// </summary>
        public short ct_temp1;

        /// <summary>
        /// 标记颜色0, bgr格式
        /// </summary>
        public int ct_color0;

        /// <summary>
        /// 标记颜色1, bgr格式
        /// </summary>
        public int ct_color1;

        public void Reset()
        {
            change_type = StrechControlFlags.SCF_All;
            strech_type = strech_type.STRECH_TYPE_PHE;
            brightness = 50;
            contrast = 50;

            gain = 2.0f;
            dde_level = 0;

            //
            man_temp0 = 200; man_temp1 = 400;
            man_enable = 0;

            //
            phe_stream = 0;

            //
            ct_type = ColorTempType.None;
            ct_temp0 = 200;
            ct_color0 = 0x000000;
            ct_temp1 = 400;
            ct_color1 = 0x000000;
        }
    }


    /// <summary>
    /// H264帧头V2, 混合流单帧数据帧头
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 1024)]
    public struct H264FrameHeaderV2
    {
        /// <summary>
        /// 大小
        /// </summary>
        public int Size;

        /// <summary>
        /// 宽度
        /// </summary>
        public ushort Width;

        /// <summary>
        /// 高度
        /// </summary>
        public ushort Height;

        /// <summary>
        /// 设备工作温度
        /// </summary>
        public int ShellTemp;

        /// <summary>
        /// 局部测温对象标志
        /// </summary>
        public int LocalResultFlags;

        /// <summary>
        /// 全局+局部测温对象
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 17)]
        public xxxresult[] ResultArray;

        public xxxmouseresult MouseResult;

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static H264FrameHeaderV2 Create()
        {
            H264FrameHeaderV2 x = new H264FrameHeaderV2();
            x.ResultArray = new xxxresult[17];
            return x;
        }
    }

    [StructLayout(LayoutKind.Sequential, Size = 8)]
    public struct xxxmouseresult
    {
        public ushort x;
        public ushort y;
        public float temp;
    }


    /// <summary>
    /// 屏蔽区
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 20)]
    public struct xxxshield
    {
        /// <summary>
        /// 序号
        /// </summary>
        public byte id;

        byte pad1;
        byte pad2;
        byte pad3;

        /// <summary>
        /// 点数据p0,p1,p2,p3
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public ushort[] xydata;

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static xxxshield Create()
        {
            xxxshield x = new xxxshield();
            x.xydata = new ushort[8];
            return x;
        }

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static xxxshield[] CreateArray(int count)
        {
            xxxshield[] a = new xxxshield[count];
            int i = 0;
            for (; i < count; i++)
            {
                a[i].xydata = new ushort[8];
            }
            return a;
        }

        /// <summary>
        /// 复制
        /// </summary>
        /// <param name="x"></param>
        public void Copy(ref xxxshield x)
        {
            id = x.id;

            ushort[] xydata2 = x.xydata;
            int i = 0;
            for (; i < 8; i++)
            {
                xydata[i] = xydata2[i];
            }
        }

        /// <summary>
        /// 修改
        /// </summary>
        public void Change(ref xxxmeasure mo)
        {
            ushort[] xydata2 = mo.xydata;
            int i = 0;
            for (; i < 8; i++)
            {
                xydata[i] = xydata2[i];
            }
        }
    }

    /// <summary>
    /// 测温对象X
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 152)]
    public struct xxxmeasure
    {
        /// <summary>
        /// 序号
        /// </summary>
        public byte id;

        /// <summary>
        /// 类型
        /// </summary>
        public xxxmeasuretype measuretype;

        /// <summary>
        /// 点数
        /// </summary>
        public byte points;

        /// <summary>
        /// 发射率, 单位0.01
        /// </summary>
        public byte emmi;

        /// <summary>
        /// 点数据: x0,y0,x1,y1,...
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 64)]
        public ushort[] xydata;

        /// <summary>
        /// 报警类型
        /// </summary>
        public xxxalarmtype alarmtype;

        /// <summary>
        /// 测温标志
        /// </summary>
        public xxxmeasureflags measureflags;

        ushort pad;

        /// <summary>
        /// 报警温度
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public float[] alarmdata;		//t0,t1,t2

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static xxxmeasure Create()
        {
            xxxmeasure n = new xxxmeasure();
            n.id = 0;
            n.measuretype = xxxmeasuretype.point;
            n.points = 0;
            n.emmi = 97;
            n.xydata = new ushort[64];

            n.alarmtype = xxxalarmtype.none;
            n.measureflags = 0;
            n.alarmdata = new float[4];
            return n;
        }

        /// <summary>
        /// 静态构造, 复制
        /// </summary>
        /// <param name="x"></param>
        /// <returns></returns>
        public static xxxmeasure Create(ref xxxmeasure x)
        {
            xxxmeasure n = new xxxmeasure();
            n.id = x.id;
            n.measuretype = x.measuretype;
            n.points = x.points;
            n.emmi = x.emmi;
            n.xydata = new ushort[64];

            n.alarmtype = x.alarmtype;
            n.measureflags = x.measureflags;
            n.alarmdata = new float[4];

            //
            int i = 0;
            for (; i < 64; i++)
            {
                n.xydata[i] = x.xydata[i];
            }
            for (i = 0; i < 4; i++)
            {
                n.alarmdata[i] = x.alarmdata[i];
            }

            return n;
        }

        /// <summary>
        /// 静态构造, 数组
        /// </summary>
        /// <param name="count">数量</param>
        /// <returns></returns>
        public static xxxmeasure[] CreateArray(int count)
        {
            xxxmeasure[] a = new xxxmeasure[count];
            int i = 0;
            for (; i < count; i++)
            {
                a[i].xydata = new ushort[64];
                a[i].alarmdata = new float[4];
            }
            return a;
        }

        /// <summary>
        /// 复制
        /// </summary>
        /// <param name="x"></param>
        public void Copy(ref xxxmeasure x)
        {
            id = x.id;
            measuretype = x.measuretype;
            points = x.points;
            emmi = x.emmi;

            int i = 0;
            for (; i < 64; i++)
            {
                xydata[i] = x.xydata[i];
            }

            alarmtype = x.alarmtype;
            measureflags = x.measureflags;
            for (i = 0; i < 4; i++)
            {
                alarmdata[i] = x.alarmdata[i];
            }
        }
    }

    /// <summary>
    /// 测温结果
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 24)]
    public struct xxxresult
    {
        /// <summary>
        /// 报警等级
        /// </summary>
        public int alarmLevel;

        /// <summary>
        /// 最高温X
        /// </summary>
        public ushort maxX;

        /// <summary>
        /// 最高温Y
        /// </summary>
        public ushort maxY;

        /// <summary>
        /// 最低温X
        /// </summary>
        public ushort minX;

        /// <summary>
        /// 最低温Y
        /// </summary>
        public ushort minY;

        /// <summary>
        /// 最高温
        /// </summary>
        public float max;

        /// <summary>
        /// 最低温
        /// </summary>
        public float min;

        /// <summary>
        /// 平均温
        /// </summary>
        public float avg;
    }

    /// <summary>
    /// 测温模板头
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 128)]
    public struct MeasureTemplateHead
    {
        /// <summary>
        /// 版本
        /// </summary>
        public int Version;

        /// <summary>
        /// 宽度
        /// </summary>
        public ushort Width;

        /// <summary>
        /// 高度
        /// </summary>
        public ushort Height;

        /// <summary>
        /// 全局屏蔽区标志
        /// </summary>
        public int ShieldFlags;

        /// <summary>
        /// 局部测温对象标志
        /// </summary>
        public int LMOFlags;
    }

    /// <summary>
    /// 测温模板
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = ConstSize)]
    public struct MeasureTemplate
    {
        /// <summary>
        /// 模板大小
        /// </summary>
        public const int ConstSize = 2872;

        /// <summary>
        /// 模板头
        /// </summary>
        public MeasureTemplateHead Head;

        /// <summary>
        /// 全局屏蔽区
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public xxxshield[] ShieldArray;

        /// <summary>
        /// 全局测温对象
        /// </summary>
        public xxxmeasure GMO;

        /// <summary>
        /// 局部测温对象
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public xxxmeasure[] LMOArray;

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static MeasureTemplate Create()
        {
            MeasureTemplate x = new MeasureTemplate();
            x.GMO = xxxmeasure.Create();
            x.ShieldArray = xxxshield.CreateArray(8);
            x.LMOArray = xxxmeasure.CreateArray(16);
            return x;
        }

        /// <summary>
        /// 复制
        /// </summary>
        /// <param name="x"></param>
        public void Copy(ref MeasureTemplate x)
        {
            Head = x.Head;

            //gmo+lmos
            GMO.Copy(ref x.GMO);
            int i = 0;
            for (; i < 16; i++)
            {
                LMOArray[i].Copy(ref x.LMOArray[i]);
            }

            //shields
            for (i = 0; i < 8; i++)
            {
                ShieldArray[i].Copy(ref x.ShieldArray[i]);
            }
        }
    }

    /// <summary>
    /// 测温模板的测量结果
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct MeasureTemplateResult
    {
        /// <summary>
        /// 全局测量结果
        /// </summary>
        public xxxresult GMR;

        /// <summary>
        /// 局部测量结果
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 16)]
        public xxxresult[] LMRArray;

        /// <summary>
        /// 静态构造
        /// </summary>
        /// <returns></returns>
        public static MeasureTemplateResult Create()
        {
            MeasureTemplateResult x = new MeasureTemplateResult();
            x.LMRArray = new xxxresult[16];
            return x;
        }
    }
}
