using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace YoseenSDKCS
{
    public static class YoseenSDK
    {
        /// <summary>
        /// 建立SDK
        /// </summary>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_InitSDK", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_InitSDK();

        /// <summary>
        /// 释放SDK
        /// </summary>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_FreeSDK", CallingConvention = CallingConvention.Cdecl)]
        public extern static void Yoseen_FreeSDK();

        /// <summary>
        /// 设定全局Socket超时, 单位ms, 默认2000ms, 取值范围[2000, 8000]
        /// </summary>
        /// <param name="connectTimeout">连接超时</param>
        /// <param name="sendTimeout">发送超时</param>
        /// <param name="recvTimeout">接收超时</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetTimeout", CallingConvention = CallingConvention.Cdecl)]
        public extern static void Yoseen_SetTimeout(int connectTimeout, int sendTimeout, int recvTimeout);

        /// <summary>
        /// 导出文本
        /// </summary>
        /// <param name="dfh">温度数据头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="fn">文件名</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_ExportText", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_ExportText(IntPtr dfh, IntPtr dfd, string fn);

        /// <summary>
        /// 登入热像仪, 成功登入会填充热像仪基本信息.
        /// 目前只验证热像仪地址, 不验证用户名和密码; 热像仪不限制登入的用户.
        /// </summary>
        /// <param name="loginInfo">登录信息</param>
        /// <param name="cameraBasicInfo">热像仪基本信息</param>
        /// <returns>&gt;=用户句柄, &lt;0 错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_Login", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_Login(ref YoseenLoginInfo loginInfo, ref CameraBasicInfo cameraBasicInfo);

        /// <summary>
        /// 登入热像仪V2, 成功登入会填充热像仪基本信息.
        /// 热像仪的端口是固化不可更改, 可以做端口映射, 让SDK使用映射后IP和端口,访问设备
        /// </summary>
        /// <param name="loginInfo">登录信息</param>
        /// <param name="cameraBasicInfo">热像仪基本信息</param>
        /// <returns>&gt;=用户句柄, &lt;0 错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_LoginV2", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_LoginV2(ref YoseenLoginInfoV2 loginInfo, ref CameraBasicInfo cameraBasicInfo);

        /// <summary>
        /// 登出热像仪
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_Logout", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_Logout(int userHandle);

        /// <summary>
        /// 保存单帧温度到文件(jpg扩展), 不需要预览的支持. 热像仪支持同一时刻处理16个保存请求,
        /// 降低了在同一时刻请求过多时, 后面请求失败的几率.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="fileName">文件名</param>
        /// <param name="withTemp">是否带温度</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SaveFrame", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SaveFrame(int userHandle, [MarshalAs(UnmanagedType.LPUTF8Str)] string fileName, int withTemp);

        /// <summary>
        ///保存单帧温度到内存, 用户提供frameFile, SDK填充frameFile的各个数据指针.
        ///frameFile的各个数据指针指向的都是内部数据缓冲区, 对同一userHandle的保存请求会更新此内部数据缓冲区的内容.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="frameFile">文件数据结构体</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SaveFrameToMem", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SaveFrameToMem(int userHandle, ref TempFrameFile frameFile);

        /// <summary>
        /// 开始预览, 一台热像仪能同时支持 "1对16的视频流预览" 和 "1对1的温度流" 预览.
        /// 视频流预览, 后端接收到的是H264编码的图像数据和特征温度数据, 带宽要求低.
        /// 温度流预览, 后端接收到的是全帧温度数据, 带宽要求高.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="previewInfo">预览信息</param>
        /// <returns>&gt;=预览句柄, &lt;0 错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_StartPreview", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_StartPreview(int userHandle, ref YoseenPreviewInfo previewInfo);

        /// <summary>
        /// 开始预览, 预览低分辨率的温度数据(小于160x120)时, 放大温度数据.
        /// 只支持取数据，不支持显示.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="previewInfo">预览信息</param>
        /// <param name="tempScale">温度数据放大, 支持2,3,4倍放大</param>
        /// <returns>&gt;=预览句柄, &lt;0 错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_StartPreviewTempScale", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_StartPreviewTempScale(int userHandle, ref YoseenPreviewInfo previewInfo, int tempScale);


        /// <summary>
        /// 开始预览, 支持翻转90, 翻转180, 翻转270, X镜像, Y镜像.
        /// 只支持取数据，不支持显示.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="previewInfo">预览信息</param>
        /// <param name="tempTransform">温度数据翻转</param>
        /// <returns>&gt;=预览句柄, &lt;0 错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_StartPreviewTempTransform", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_StartPreviewTempTransform(int userHandle, ref YoseenPreviewInfo previewInfo, int tempTransform);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_StartPreviewRtsp", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_StartPreviewRtsp(ref YoseenRtspInfo rtspInfo, ref YoseenPreviewInfo previewInfo);

        /// <summary>
        /// 停止预览
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_StopPreview", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_StopPreview(int previewHandle);

        /// <summary>
        /// 暂停预览
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <param name="pause">0-非暂停, 1-暂停</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_PausePreview", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_PausePreview(int previewHandle, int pause);

        /// <summary>
        /// 开始保存, 温度流预览支持保存.stream和.mp4文件, 视频流预览支持保存.mp4文件.
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <param name="fileName">文件名</param>
        /// <param name="fileType">文件类型</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_BeginSave", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_BeginSave(int previewHandle, [MarshalAs(UnmanagedType.LPUTF8Str)] string fileName, int fileType);

        /// <summary>
        /// 开始保存, 温度流预览保存.stream, timeDelta>0 每隔多少帧存一帧; timeDelta<0 每隔多少秒存一帧
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <param name="fileName">文件名</param>
        /// <param name="timeDelta">时间间隔[-100,100]</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_BeginSaveStream", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_BeginSaveStream(int previewHandle, [MarshalAs(UnmanagedType.LPUTF8Str)] string fileName, int timeDelta);

        /// <summary>
        /// 停止保存
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <param name="fileType">文件类型</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_EndSave", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_EndSave(int previewHandle, int fileType);

        /// <summary>
        /// 设置温度流预览, 温度转位图算法参数; 视频流预览使用Yoseen_SetTvoutInfo.
        /// </summary>
        /// <param name="previewHandle">预览句柄</param>
        /// <param name="strechControl">温度转位图算法参数</param>
        /// <param name="paletteType">调色板</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_PreviewSetImage", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_PreviewSetImage(int previewHandle, ref strech_control strechControl, int paletteType);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_PreviewEnableMask", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_PreviewEnableMask(int previewHandle, int enable);

        /// <summary>
        /// 发现热像仪
        /// </summary>
        /// <param name="discoverFlags">0x01广播, 0x02组播, 0x04 广播2</param>
        /// <returns>响应链表</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DiscoverCameras", CallingConvention = CallingConvention.Cdecl)]
        public extern static IntPtr Yoseen_DiscoverCameras(int discoverFlags);

        /// <summary>
        /// 释放响应链表
        /// </summary>
        /// <param name="pp">响应链表</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DiscoverCamerasFree", CallingConvention = CallingConvention.Cdecl)]
        public extern static void Yoseen_DiscoverCamerasFree(ref IntPtr pp);

        public static DiscoverCameraResp2[] Yoseen_DiscoverCameras2(int discoverFlags)
        {
            List<DiscoverCameraResp2> result = new List<DiscoverCameraResp2>();
            IntPtr pHead = Yoseen_DiscoverCameras(discoverFlags);
            IntPtr p = pHead;
            while (p != IntPtr.Zero)
            {
                DiscoverCameraResp2 dcr = (DiscoverCameraResp2)Marshal.PtrToStructure(p, typeof(DiscoverCameraResp2));
                result.Add(dcr);
                p = dcr.pNext;
            }

            if (pHead != IntPtr.Zero)
            {
                Yoseen_DiscoverCamerasFree(ref pHead);
            }
            return result.ToArray();
        }

        /// <summary>
        /// 获取热像仪基本信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraBasicInfo">基本信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetCameraBasicInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetCameraBasicInfo(int userHandle, ref CameraBasicInfo cameraBasicInfo);

        /// <summary>
        /// 设置热像仪基本信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraBasicInfo">基本信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetCameraBasicInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetCameraBasicInfo(int userHandle, ref CameraBasicInfo cameraBasicInfo);

        /// <summary>
        /// 获取热像仪网络信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraNetworkInfo">网络信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetCameraNetworkInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetCameraNetworkInfo(int userHandle, ref CameraNetworkInfo cameraNetworkInfo);

        /// <summary>
        /// 设置热像仪网络信息, 成功设置后, 热像仪IP可能发生变化, 请重新发现热像仪.
        /// 若配置成使用静态IP, 但是忘记IP, 请使用热像仪的GPIO复位热像仪.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraNetworkInfo">网络信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetCameraNetworkInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetCameraNetworkInfo(int userHandle, ref CameraNetworkInfo cameraNetworkInfo);

        /// <summary>
        /// 获取探测器配置信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="fpaInfo">探测器配置信息</param>
        /// <param name="index">档位</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetFPAInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetFPAInfo(int userHandle, ref FPAInfo fpaInfo, int index);

        /// <summary>
        /// 设置探测器配置信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="fpaInfo">探测器配置信息</param>
        /// <param name="index">档位</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetFPAInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetFPAInfo(int userHandle, ref FPAInfo fpaInfo, int index);

        /// <summary>
        /// 获取挡板校零信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="ffcInfo">挡板校零信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetFFCInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetFFCInfo(int userHandle, ref FFCInfo ffcInfo);

        /// <summary>
        /// 设置挡板校零信息, 设置成功后, 会触发一次挡板校零.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="ffcInfo">挡板校零信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetFFCInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetFFCInfo(int userHandle, ref FFCInfo ffcInfo);

        /// <summary>
        /// 获取模拟视频信息, 模拟视频信息也包含图像算法参数\流媒体参数.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="tvoutInfo">模拟视频信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetTvoutInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetTvoutInfo(int userHandle, ref TvoutInfo tvoutInfo);

        /// <summary>
        /// 设置模拟视频信息, 模拟视频信息也包含图像算法参数\流媒体参数.
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="tvoutInfo">模拟视频信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetTvoutInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetTvoutInfo(int userHandle, ref TvoutInfo tvoutInfo);

        /// <summary>
        /// 获取测温修正信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="measureInfo">测温修正信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetFixInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetFixInfo(int userHandle, ref FixInfo fixInfo);

        /// <summary>
        /// 设置测温修正信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="measureInfo">测温修正信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetFixInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetFixInfo(int userHandle, ref FixInfo fixInfo);

        /// <summary>
        /// 获取热像仪时间信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraTimeInfo">时间信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetCameraTimeInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetCameraTimeInfo(int userHandle, ref CameraTimeInfo cameraTimeInfo);

        /// <summary>
        /// 设置热像仪时间信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="cameraTimeInfo">时间信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetCameraTimeInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetCameraTimeInfo(int userHandle, ref CameraTimeInfo cameraTimeInfo);

        /// <summary>
        /// 获取GPIO信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="gpioInfo">GPIO信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetGpioInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetGpioInfo(int userHandle, ref GpioInfo gpioInfo);

        /// <summary>
        /// 设置GPIO信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="gpioInfo">GPIO信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetGpioInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetGpioInfo(int userHandle, ref GpioInfo gpioInfo);

        /// <summary>
        /// 获取串口信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="serialPortInfo">串口信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetSerialPortInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetSerialPortInfo(int userHandle, ref SerialPortInfo serialPortInfo);

        /// <summary>
        /// 设置串口信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="serialPortInfo">串口信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetSerialPortInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetSerialPortInfo(int userHandle, ref SerialPortInfo serialPortInfo);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetFPAInfoHK160", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetFPAInfoHK160(int userHandle, ref FPAInfoHK160 info);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetFPAInfoHK160", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetFPAInfoHK160(int userHandle, ref FPAInfoHK160 info);

        /// <summary>
        /// 发送控制信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="ctl">控制信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SendControl", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SendControl(int userHandle, ref Ctl ctl);

        /// <summary>
        /// 发送控制信息
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="ctl">控制信息</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SendControl", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SendControl(int userHandle, IntPtr ctl);

        /// <summary>
        /// 发送控制信息X
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="ctlx">控制信息X</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SendControlX", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SendControlX(int userHandle, ref CtlX ctlx);

        /// <summary>
        /// 发送串口数据到热像仪, 热像仪接收到数据, 直接写到串口
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="spd">串口数据</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_WriteSerialPortData", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_WriteSerialPortData(int userHandle, ref xxxserialportdata spd);

        /// <summary>
        /// 上传本地文件到热像仪, 用于热像仪程序更新、坏点补偿等
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="fn">文件名称</param>
        /// <param name="fileType">文件类型</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_UploadFile", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_UploadFile(int userHandle, string fn, int fileType);

        /// <summary>
        /// 下载热像仪文件到本地, 用于热像仪日志分析等
        /// </summary>
        /// <param name="userHandle">用户句柄</param>
        /// <param name="fn">文件名称</param>
        /// <param name="fileType">文件类型</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DownloadFile", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_DownloadFile(int userHandle, string fn, int fileType);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_GetCameraOutputInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_GetCameraOutputInfo(int userHandle, ref CameraOutputInfo info);
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_SetCameraOutputInfo", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_SetCameraOutputInfo(int userHandle, ref CameraOutputInfo info);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_UploadMem", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_UploadMem(int userHandle, int memType, IntPtr memData, int memSize);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DownloadMem", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_DownloadMem(int userHandle, int memType, IntPtr memData, ref int memSize);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DiscoverConfigIP", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_DiscoverConfigIP(ref ConfigIPReq req);

        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_DiscoverConfigIP", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Yoseen_DiscoverConfigIP2(IntPtr req);
    }

    public static class YoseenPlayback
    {
        public const int JpgDataMaxSize = 256 + 1280 * 1024 * 4;

        /// <summary>
        /// 建立回放上下文
        /// </summary>
        /// <returns>回放上下文</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_Create", CallingConvention = CallingConvention.Cdecl)]
        public extern static IntPtr YoseenPlayback_Create();

        /// <summary>
        /// 释放回放上下文
        /// </summary>
        /// <param name="pp">回放上下文</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_Free", CallingConvention = CallingConvention.Cdecl)]
        public extern static void YoseenPlayback_Free(ref IntPtr pp);

        /// <summary>
        /// 设置回放图像算法参数
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="sc">算法参数</param>
        /// <param name="paletteType">调色板类型</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_GetImage", CallingConvention = CallingConvention.Cdecl)]
        public extern static void YoseenPlayback_GetImage(IntPtr ctx, ref strech_control sc, ref int paletteType);

        /// <summary>
        /// 获取回放图像算法参数
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="sc">算法参数</param>
        /// <param name="paletteType">调色板类型</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_SetImage", CallingConvention = CallingConvention.Cdecl)]
        public extern static void YoseenPlayback_SetImage(IntPtr ctx, ref strech_control sc, ref int paletteType);

        /// <summary>
        /// 打开文件, 支持单帧bmp,png,jpg; 多帧stream文件
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="fileName">文件名</param>
        /// <param name="fileType">文件类型</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_OpenFile", CallingConvention = CallingConvention.Cdecl)]
        public extern static int YoseenPlayback_OpenFile(IntPtr ctx, string fileName, int fileType);

        /// <summary>
        /// 打开内存
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="dfh">温度数据帧头</param>
        /// <param name="dfd">温度数据</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_OpenMem", CallingConvention = CallingConvention.Cdecl)]
        public extern static int YoseenPlayback_OpenMem(IntPtr ctx, IntPtr dfh, IntPtr dfd);

        /// <summary>
        /// 关闭文件
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_CloseFile", CallingConvention = CallingConvention.Cdecl)]
        public extern static void YoseenPlayback_CloseFile(IntPtr ctx);

        /// <summary>
        /// 单帧文件, 保存数据.
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="tff">封面数据</param>
        /// <param name="fileName">文件名</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_SaveFile", CallingConvention = CallingConvention.Cdecl)]
        public extern static int YoseenPlayback_SaveFile(IntPtr ctx, ref TempFrameFile tff, string fileName);

        /// <summary>
        /// 多帧文件, 读取流信息
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="tff">封面数据</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_ReadCover", CallingConvention = CallingConvention.Cdecl)]
        public extern static int YoseenPlayback_ReadCover(IntPtr ctx, ref TempFrameFile tff);

        /// <summary>
        /// 读取温度数据. 每次读取一帧, 位图数据重新生成.
        /// </summary>
        /// <param name="ctx">回放上下文</param>
        /// <param name="frameIndex">帧序号</param>
        /// <param name="dataFrame">温度帧</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "YoseenPlayback_ReadFrame", CallingConvention = CallingConvention.Cdecl)]
        public extern static int YoseenPlayback_ReadFrame(IntPtr ctx, int frameIndex, ref DataFrame dataFrame);
    }

    public static class YoseenAlg
    {
        /// <summary>
        /// 测量温度
        /// </summary>
        /// <param name="mea">测温对象</param>
        /// <param name="dfh">温度数据头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="result">测量结果</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtGetResult", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtGetResult(IntPtr dfh, IntPtr dfd, ref xxxmeasure mea, ref xxxresult result);

        /// <summary>
        /// 测量全局测温
        /// </summary>
        /// <param name="dfh">温度数据头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="result">测量结果</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtGetResult_GMO", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtGetResult_GMO(IntPtr dfh, IntPtr dfd, ref xxxresult result);

        /// <summary>
        /// 建立测温模板上下文
        /// </summary>
        /// <param name="width">数据宽度</param>
        /// <param name="height">数据高度</param>
        /// <returns>模板上下文</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtCreate", CallingConvention = CallingConvention.Cdecl)]
        public extern static IntPtr mtCreate(int width, int height);

        /// <summary>
        /// 释放测温模板
        /// </summary>
        /// <param name="pp">模板上下文</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtFree", CallingConvention = CallingConvention.Cdecl)]
        public extern static void mtFree(ref IntPtr pp);

        /// <summary>
        /// 重载测温模板, 指针
        /// </summary>
        /// <param name="context">模板上下文</param>
        /// <param name="ptrMT">测温模板</param>
        /// <returns>0成功,x失败</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtReload", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtReload(IntPtr context, IntPtr ptrMT);

        /// <summary>
        /// 重载测温模板, 结构体
        /// </summary>
        /// <param name="context">模板上下文</param>
        /// <param name="mt">测温模板</param>
        /// <returns>0成功,x失败</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtReload", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtReload2(IntPtr context, ref MeasureTemplate mt);

        /// <summary>
        /// 获取测温模板的测量结果, 指针
        /// </summary>
        /// <param name="context">模板上下文</param>
        /// <param name="dfh">温度数据帧头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="ptrMTR">测量结果</param>
        /// <returns>0成功,x失败</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtCheck", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtCheck(IntPtr context, IntPtr dfh, IntPtr dfd, IntPtr ptrMTR);

        /// <summary>
        /// 获取测温模板的测量结果, 结构体
        /// </summary>
        /// <param name="context">模板上下文</param>
        /// <param name="dfh">温度数据帧头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="mtr">测量结果</param>
        /// <returns>0成功,x失败</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "mtCheck", CallingConvention = CallingConvention.Cdecl)]
        public extern static int mtCheck2(IntPtr context, IntPtr dfh, IntPtr dfd, ref MeasureTemplateResult mtr);


        /// <summary>
        /// 检查校准表
        /// </summary>
        /// <param name="filename">文件名</param>
        /// <param name="camera_id">热像仪序列号</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "caliVerify", CallingConvention = CallingConvention.Cdecl)]
        public extern static int caliVerify(string filename, string camera_id);

        /// <summary>
        /// 检查坏点表
        /// </summary>
        /// <param name="filename">文件名</param>
        /// <param name="bfh">坏点文件头</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "caliVerifyBad", CallingConvention = CallingConvention.Cdecl)]
        public extern static int caliVerifyBad(string filename, ref bad_file_header bfh);

        /// <summary>
        /// 检查二次分段校准表
        /// </summary>
        /// <param name="filename">文件名</param>
        /// <param name="camera_id">热像仪序列号</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "caliVerifyC2m", CallingConvention = CallingConvention.Cdecl)]
        public extern static int caliVerifyC2m(string filename, string camera_id);

        /// <summary>
        /// 检查用户校准文件
        /// </summary>
        /// <param name="filename">文件名</param>
        /// <param name="camera_id">热像仪序列号</param>
        /// <returns></returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "caliVerifyCust", CallingConvention = CallingConvention.Cdecl)]
        public extern static int caliVerifyCust(string filename, string camera_id);

        /// <summary>
        /// 分配调色板数据
        /// </summary>
        /// <param name="paletteCount">调色板数量</param>
        /// <returns>调色板数据</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_AllocPaletteData", CallingConvention = CallingConvention.Cdecl)]
        public extern static IntPtr Yoseen_AllocPaletteData(ref int paletteCount);

        /// <summary>
        /// 释放调色板数据
        /// </summary>
        /// <param name="pp">调色板数据</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "Yoseen_FreePaletteData", CallingConvention = CallingConvention.Cdecl)]
        public extern static void Yoseen_FreePaletteData(ref IntPtr pp);

        /// <summary>
        /// 放大温度数据, 双线性插值方式
        /// </summary>
        /// <param name="scale">放大倍数, 支持2,3,4倍放大</param>
        /// <param name="srcWidth">源宽度</param>
        /// <param name="srcHeight">源高度</param>
        /// <param name="srcData">源数据</param>
        /// <param name="dstData">目的数据</param>
        /// <returns>错误码</returns>
        [DllImport("YoseenSDK.dll", EntryPoint = "Alg_ScaleTemp", CallingConvention = CallingConvention.Cdecl)]
        public extern static int Alg_ScaleTemp(int scale, ushort srcWidth, ushort srcHeight, IntPtr srcData, IntPtr dstData);

        /// <summary>
        /// 数据翻转
        /// </summary>
        /// <param name="dfh">温度帧头</param>
        /// <param name="dfd">温度数据区</param>
        /// <param name="tmp">辅助数据区(90度翻转需要)</param>
        /// <param name="transformType">翻转类型</param>
        [DllImport("YoseenSDK.dll", EntryPoint = "Alg_Transform", CallingConvention = CallingConvention.Cdecl)]
        public extern static void Alg_Transform(IntPtr dfh, IntPtr dfd, IntPtr tmp, int transformType);
    }
}
