using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace YoseenSDKCS
{
    /// <summary>
    /// 发现热像仪标志
    /// </summary>
    public enum xxxdiscoverflags
    {
        /// <summary>
        /// 广播方式
        /// </summary>
        xxxdiscoverflags_broadcast = 0x0001,

        /// <summary>
        /// 组播方式
        /// </summary>
        xxxdiscoverflags_multicast = 0x0002,

        /// <summary>
        /// 广播方式2
        /// </summary>
        xxxdiscoverflags_broadcast2 = 0x0004,
    }

    /// <summary>
    /// 发现热像仪响应2
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct DiscoverCameraResp2
    {
        /// <summary>
        /// 热像仪基本信息
        /// </summary>
        public CameraBasicInfo BasicInfo;

        /// <summary>
        /// 热像仪IP
        /// </summary>
        public UInt32 CameraIp;

        /// <summary>
        /// 下一个响应
        /// </summary>
        public IntPtr pNext;
    }

    /// <summary>
    /// 热像仪基本信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Size = 216)]
    public struct CameraBasicInfo
    {
        /// <summary>
        /// 热像仪序列号
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string CameraId;

        /// <summary>
        /// 热像仪名称
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string CameraName;

        /// <summary>
        /// 热像仪类型
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 16)]
        public string CameraType;

        /// <summary>
        /// 探测器序列号
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string FPAId;

        /// <summary>
        /// 硬件版本
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string HardwareId;

        /// <summary>
        /// 软件版本
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string SoftwareId;

        /// <summary>
        /// 数据宽度
        /// </summary>
        public UInt16 DataWidth;

        /// <summary>
        /// 数据高度
        /// </summary>
        public UInt16 DataHeight;

        /// <summary>
        /// 数据帧率
        /// </summary>
        public UInt16 DataFps;

        /// <summary>
        /// 数据率, 温度数据每隔多少帧使用一帧
        /// </summary>
        public UInt16 DataRatio;

        /// <summary>
        /// 发送率, 温度数据每隔多少帧发送一帧, 最终用户接收到的温度数据的帧率是DataFps/(DataRatio*SendRatio)
        /// </summary>
        public UInt16 SendRatio;

        /// <summary>
        /// 数据变换类型, 支持前端数据旋转和镜像
        /// </summary>
        public byte DataTransform;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 25)]
        byte[] pad;

        public ushort VideoWidth;
        public ushort VideoHeight;

        public override string ToString()
        {
            return string.Format("{0}, {1}, {2}, {3}, {4}-{5}-{6}\n",
                CameraId, CameraType, FPAId, SoftwareId,
                DataRatio, SendRatio, DataTransform);
        }
    }

    /// <summary>
    /// 热像仪网络信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 52)]
    public struct CameraNetworkInfo
    {
        /// <summary>
        /// 是否使用静态IP
        /// </summary>
        [MarshalAs(UnmanagedType.I1)]
        public bool UseStaticIp;

        Byte pad1;

        /// <summary>
        /// 组播端口, 未使用
        /// </summary>
        public UInt16 MulticastPort;

        /// <summary>
        /// 静态IP
        /// </summary>
        public UInt32 StaticIp;

        /// <summary>
        /// 子网掩码
        /// </summary>
        public UInt32 SubnetMask;

        /// <summary>
        /// 网关
        /// </summary>
        public UInt32 Gateway;

        /// <summary>
        /// 组播IP, 未使用
        /// </summary>
        public UInt32 MulticastIp;

        /// <summary>
        /// MAC地址
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 6)]
        public Byte[] MacAddr;

        Byte pad2;
        Byte pad3;

        /// <summary>
        /// 主DNS
        /// </summary>
        public UInt32 Dns;

        /// <summary>
        /// 次DNS
        /// </summary>
        public UInt32 Dns2;

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <returns>热像仪网络信息</returns>
        public static CameraNetworkInfo create()
        {
            CameraNetworkInfo a = new CameraNetworkInfo();
            a.MacAddr = new byte[6];
            return a;
        }

        public override string ToString()
        {
            string macStr = string.Format("{0:x2}:{1:x2}:{2:x2}:{3:x2}:{4:x2}:{5:x2}",
                MacAddr[0], MacAddr[1], MacAddr[2],
                MacAddr[3], MacAddr[4], MacAddr[5]);

            return string.Format("{0}; {1}, {2}, {3}; {4},{5}; {6}\n",
                UseStaticIp ? "static" : "dhcp",
                YoseenUtil.uint2str(StaticIp), YoseenUtil.uint2str(SubnetMask), YoseenUtil.uint2str(Gateway),
                YoseenUtil.uint2str(Dns), YoseenUtil.uint2str(Dns2),
                macStr);
        }
    }

    /// <summary>
    /// 探测器配置参数
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct FPAInfo
    {
        public const double AD5625R_SENSITIVITY = 5.0 / 4096.0;

        public UInt16 gsk;
        public UInt16 gfid;
        public UInt16 vsk;
        public UInt16 vbus;

        public UInt16 tint;
        public xxxfpacint cint;
        Byte pad;

        public override string ToString()
        {
            double gsk2 = gsk * AD5625R_SENSITIVITY;
            double gfid2 = gfid * AD5625R_SENSITIVITY;
            double vsk2 = vsk * AD5625R_SENSITIVITY;
            double vbus2 = vbus * AD5625R_SENSITIVITY;
            return string.Format("gsk {0:F3}, gfid {1:F3}, vsk {2:F3}, vbus {3:F3}, tint {4}, cint {5}\n",
                gsk2, gfid2, vsk2, vbus2,
                tint, (byte)cint);
        }
    }

    /// <summary>
    /// 挡板校零信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 16)]
    public struct FFCInfo
    {
        /// <summary>
        /// 挡板关闭后忽略帧数, 用户不可改
        /// </summary>
        public Byte frames_skipped_after_close;

        /// <summary>
        /// 挡板关闭时累积帧数, 用户不可改
        /// </summary>
        public Byte frames_accumlated_when_closed;

        /// <summary>
        /// 挡板打开后忽略帧数, 用户不可改
        /// </summary>
        public Byte frames_skipped_after_open;

        /// <summary>
        /// 无挡片模式开关
        /// </summary>
        public byte nst_enable;

        /// <summary>
        /// 挡板校零时间间隔
        /// </summary>
        public UInt16 time_interval;

        /// <summary>
        /// 挡板校零温度间隔
        /// </summary>
        public UInt16 temp_interval;

        /// <summary>
        /// 自动对焦回弹时间
        /// </summary>
        public sbyte af_rebound;

        /// <summary>
        /// 有挡片1/(1C/60s), 内部参数
        /// </summary>
        public byte fpa_shutter;

        /// <summary>
        /// 无挡片模式温漂
        /// </summary>
        public short nst_delta;
    }

    /// <summary>
    /// 测温修正信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 72)]
    public struct FixInfo
    {
        /// <summary>
        /// 气温
        /// </summary>
        public float AtmosphericTemperature;

        /// <summary>
        /// 相对湿度
        /// </summary>
        public float RelativeHumidity;

        /// <summary>
        /// 能见度
        /// </summary>
        public float Visibility;

        /// <summary>
        /// 降雨强度
        /// </summary>
        public float RainfallIntensity;

        /// <summary>
        /// 降雪强度
        /// </summary>
        public float SnowfallIntensity;

        /// <summary>
        /// 目标距离
        /// </summary>
        public float TargetDistance;

        /// <summary>
        /// 全局发射率
        /// </summary>
        public float GlobalEmissivity;

        /// <summary>
        /// 红外窗口透过率
        /// </summary>
        public float InfraredWindowTrans;

        /// <summary>
        /// 温度漂移
        /// </summary>
        public float TempOffset;

        /// <summary>
        /// 是否开启大气修正
        /// </summary>
        public byte EnableAtmFix;
    }

    public enum XuiDisplayFlags
    {
        /// <summary>
        /// 显示调色板
        /// </summary>
        XDF_Palette = 0x0001,

        /// <summary>
        /// 显示高温追踪
        /// </summary>
        XDF_TrackHigh = 0x0002,

        /// <summary>
        /// 显示低温追踪
        /// </summary>
        XDF_TrackLow = 0x0004,

        /// <summary>
        /// 不显示全局最高温
        /// </summary>
        XDF_NoGlobalMax = 0x0008,

        /// <summary>
        /// 不显示全局最低温
        /// </summary>
        XDF_NoGlobalMin = 0x0010,

        /// <summary>
        /// 不显示局部测温对象
        /// </summary>
        XDF_NoLocalMeasures = 0x0020,
    };

    /// <summary>
    /// HDMI
    /// </summary>
    public enum HDMIPreset : byte
    {
        HDMIPreset_720x576 = 0,
        HDMIPreset_1280x720,
        HDMIPreset_1280x720_960x720,
    };

    /// <summary>
    /// 模拟视频信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 40)]
    public struct TvoutInfo
    {
        /// <summary>
        /// 开启模拟视频
        /// </summary>
        [MarshalAs(UnmanagedType.I1)]
        public bool EnableTvout;

        /// <summary>
        /// 调色板
        /// </summary>
        public xxxpalette PaletteType;

        byte TvStdId;

        /// <summary>
        /// 对比度
        /// </summary>
        public Byte Contrast;

        /// <summary>
        /// 亮度
        /// </summary>
        public Byte Brightness;

        /// <summary>
        /// 数字变焦
        /// </summary>
        public byte Zoom;

        /// <summary>
        /// 显示标志位, XuiDisplayFlags
        /// </summary>
        public ushort XuiDisplayFlags;

        /// <summary>
        /// 增益
        /// </summary>
        public float Gain;

        /// <summary>
        /// 流媒体数据量大小, 单位bps
        /// </summary>
        public uint H264_Bitrate;

        /// <summary>
        /// 流媒体图像组大小
        /// </summary>
        public ushort H264_GopSize;

        /// <summary>
        /// 算法类型
        /// </summary>
        public strech_type StrechType;

        /// <summary>
        /// DDE等级
        /// </summary>
        public byte DDELevel;

        /// <summary>
        /// 手动温度0, 单位0.1C
        /// </summary>
        public short man_temp0;

        /// <summary>
        /// 手动温度1, 单位0.1C
        /// </summary>
        public short man_temp1;

        /// <summary>
        /// 手动温度
        /// </summary>
        public byte man_enable;

        byte pad;

        /// <summary>
        /// HDMI分辨率
        /// </summary>
        public HDMIPreset HMDI;

        /// <summary>
        /// 温度标记类型
        /// 
        /// 高温:	温度高于ct_temp0, 用ct_color0
        /// 低温：	温度低于ct_temp0, 用ct_color0
        /// 区间内： 温度介于[ct_temp0, ct_temp1], 用ct_color0
        /// 区间外:	温度低于ct_temp0, 用ct_color0; 温度高于ct_temp1,用ct_color1
        /// 
        /// </summary>
        public ColorTempType CT_Type;

        /// <summary>
        /// 标记温度0, 单位0.1C
        /// </summary>
        public short CT_Temp0;

        /// <summary>
        /// 标记温度1, 单位0.1C
        /// </summary>
        public short CT_Temp1;

        /// <summary>
        /// 标记颜色0
        /// </summary>
        public int CT_Color0;

        /// <summary>
        /// 标记颜色1
        /// </summary>
        public int CT_Color1;
    }

    /// <summary>
    /// 串口信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 16)]
    public struct SerialPortInfo
    {
        /// <summary>
        /// 用法
        /// </summary>
        public xxxspusage Usage;

        /// <summary>
        /// PELCO地址
        /// </summary>
        public byte PelcoAddr;

        /// <summary>
        /// 波特率
        /// </summary>
        public xxxspbaudrate BaudRate;

        /// <summary>
        /// 数据位
        /// </summary>
        public xxxspdatabits DataBits;

        /// <summary>
        /// 停止位
        /// </summary>
        public xxxspstopbits StopBits;

        /// <summary>
        /// 校验位
        /// </summary>
        public xxxspparity Parity;
    }

    /// <summary>
    /// GPIO信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 16)]
    public struct GpioInfo
    {
        /// <summary>
        /// 输入0
        /// </summary>
        public xxxgpioinputtype Input0;

        /// <summary>
        /// 输入1
        /// </summary>
        public xxxgpioinputtype Input1;

        /// <summary>
        /// 输出0
        /// </summary>
        public xxxgpiooutputtype Output0;

        /// <summary>
        /// 输出1
        /// </summary>
        public xxxgpiooutputtype Output1;
    }

    /// <summary>
    /// 热像仪时间信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 128)]
    public struct CameraTimeInfo
    {
        /// <summary>
        /// 主时间服务器
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string NtpServer;

        /// <summary>
        /// 次时间服务器
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string NtpServer2;

        /// <summary>
        /// 时区
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string TimeZone;
    }

    /// <summary>
    /// 控制类型
    /// </summary>
    public enum CtlType : ushort
    {
        /// <summary>
        /// 测试, 内部使用
        /// </summary>
        CtlType_Debug = 0,

        /// <summary>
        /// 修改数据类型, 内部使用
        /// </summary>
        CtlType_ChangeDataType,

        /// <summary>
        /// 动挡板, 内部使用
        /// </summary>
        CtlType_MoveShutter,

        /// <summary>
        /// 保存出厂配置, 内部使用
        /// </summary>
        CtlType_StoreFactory,

        /// <summary>
        /// 重启热像仪
        /// </summary>
        CtlType_RestartCamera,

        /// <summary>
        /// 手动挡板校零
        /// </summary>
        CtlType_ManualFFC = 32,

        /// <summary>
        /// 重置出厂配置
        /// </summary>
        CtlType_RestoreFactory,

        /// <summary>
        /// 手动对焦
        /// </summary>
        CtlType_FocusManual,

        /// <summary>
        /// 自动对焦
        /// </summary>
        CtlType_FocusAuto,

        /// <summary>
        /// 设置报警, 拉高或拉低热像仪GPIO输出管脚
        /// </summary>
        CtlType_SetAlarm,

        /// <summary>
        /// 禁用挡板校零, 在一段时间内禁用挡板校零, 保证温度数据不因校零而丢失
        /// </summary>
        CtlType_DisableFFC,

        /// <summary>
        /// 启停模式控制电动镜头电机
        /// </summary>
        CtlType_FocusManual2,
    }

    /// <summary>
    /// GPIO报警
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct GpioAlarm
    {
        /// <summary>
        /// 报警值0
        /// </summary>
        public xxxgpioalarmvalue value0;

        /// <summary>
        /// 报警方式0
        /// </summary>
        public xxxgpioalarmtype type0;

        /// <summary>
        /// 报警时间0
        /// </summary>
        public byte time0;

        /// <summary>
        /// 报警值1
        /// </summary>
        public xxxgpioalarmvalue value1;

        /// <summary>
        /// 报警方式1
        /// </summary>
        public xxxgpioalarmtype type1;

        /// <summary>
        /// 报警时间1
        /// </summary>
        public byte time1;
    }

    /// <summary>
    /// 控制信息
    /// </summary>
    [StructLayout(LayoutKind.Explicit, Size = 10)]
    public struct Ctl
    {
        /// <summary>
        /// 控制类型
        /// </summary>
        [FieldOffset(0)]
        public CtlType Type;

        /// <summary>
        /// 挡板状态, 0-开, 1-关
        /// </summary>
        [FieldOffset(2)]
        public Byte ShutterState;

        /// <summary>
        /// 数据类型, 内部使用
        /// </summary>
        [FieldOffset(2)]
        public byte DataType;

        /// <summary>
        /// 电动镜头电机转动时间, 单位5ms, 正数-远焦, 负数-近焦
        /// </summary>
        [FieldOffset(2)]
        public Int16 FocusDelta;

        /// <summary>
        /// 自动对焦信息
        /// </summary>
        [FieldOffset(2)]
        public xxxfocusrect FocusRect;

        /// <summary>
        /// GPIO报警
        /// </summary>
        [FieldOffset(2)]
        public GpioAlarm Alarm;

        /// <summary>
        /// 禁用挡板校零, 0-不禁用, 1-禁用
        /// </summary>
        [FieldOffset(2)]
        public byte DisableFFC;

        /// <summary>
        /// 电动镜头电机转动方式, 0 停止, 1 远焦转动, 2 近焦转动
        /// </summary>
        [FieldOffset(2)]
        public byte FocusType;
    }

    /// <summary>
    /// 控制X类型
    /// </summary>
    public enum CtlXType : int
    {
        /// <summary>
        /// 获取设备时间, UTC
        /// </summary>
        CtlXType_GetTime = 0,

        /// <summary>
        /// 设置设备仪时间, UTC
        /// </summary>
        CtlXType_SetTime,

        /// <summary>
        /// 获取测温档位
        /// </summary>
        CtlXType_GetGear,

        /// <summary>
        /// 设置测温档位
        /// </summary>
        CtlXType_SetGear,

        /// <summary>
        /// 自动对焦
        /// </summary>
        CtlXType_AutoFocus,

        /// <summary>
        /// 位置记忆电动镜头, 获取配置
        /// </summary>
        CtlXType_XfGetConfig,

        /// <summary>
        /// 位置记忆电动镜头, 设置配置, 内部使用
        /// </summary>
        CtlXType_XfSetConfig,

        /// <summary>
        /// 位置记忆电动镜头, 获取位置
        /// </summary>
        CtlXType_XfGetCur,

        /// <summary>
        /// 位置记忆电动镜头, 设置位置
        /// </summary>
        CtlXType_XfMove,

        /// <summary>
        /// 位置记忆电动镜头, 重置位置
        /// </summary>
        CtlXType_XfReset,

        /// <summary>
        /// 位置记忆电动镜头, 检测过冲
        /// </summary>
        CtlXType_XfCheckDelta,

        /// <summary>
        /// 获取外置黑体配置
        /// </summary>
        CtlXType_GetExtBbConfig,

        /// <summary>
        /// 设置外置黑体配置
        /// </summary>
        CtlXType_SetExtBbConfig,

        /// <summary>
        /// 
        /// </summary>
        CtlXType_SyncNTP,

        /// <summary>
        /// 
        /// </summary>
        CtlXType_ExecutePtz,

        /// <summary>
        /// 
        /// </summary>
        CtlXType_SetMouseMea,


        /// <summary>
        /// 
        /// </summary>
        CtlXType_SetGpioOutput = 128,


        //oem,hwyz,20191218
        CtlXType_OemHWYZ_GetData = 256,
        CtlXType_OemHWYZ_GetDelta,
        CtlXType_OemHWYZ_SetDelta,
    }

    /// <summary>
    /// 外置黑体配置
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct ExtBbConfig
    {
        /// <summary>
        /// 开关
        /// </summary>
        public ushort enable;

        /// <summary>
        /// 黑体温度, 单位: 0.1C
        /// </summary>
        public short temp;

        /// <summary>
        /// 黑体中心X
        /// </summary>
        public ushort x;

        /// <summary>
        /// 黑体中心Y
        /// </summary>
        public ushort y;

        /// <summary>
        /// 黑体有效半径, 保持1 
        /// </summary>
        public ushort radius;
    }

    [StructLayout(LayoutKind.Sequential, Size = 8)]
    public struct xxxgpiooutput
    {
        #region static
        public const int ConstFlags_Light = 1 << 0;
        public const int ConstFlags_Ledy = 1 << 1;
        public const int ConstFlags_Ledb = 1 << 2;
        public const int ConstFlags_Alarm = 1 << 3;
        public const int ConstFlags_Buzz = 1 << 4;
        #endregion

        public int flags;
        public int value;
    }

    [StructLayout(LayoutKind.Sequential, Size = 16)]
    public struct xxxptzcmd
    {
        public byte act;
        public byte arg;
        short pad;

        //
        public int pan;
        public int tilt;
        public int zoom;
    }

    [StructLayout(LayoutKind.Sequential, Size = 8)]
    public struct xxxmousemea
    {
        public float x;
        public float y;
    }

    /// <summary>
    /// 控制信息X
    /// </summary>
    [StructLayout(LayoutKind.Explicit, Size = 68)]
    public struct CtlX
    {
        /// <summary>
        /// 控制类型X
        /// </summary>
        [FieldOffset(0)]
        public CtlXType Type;

        /// <summary>
        /// 错误码
        /// </summary>
        [FieldOffset(4)]
        public int Error;

        /// <summary>
        /// 设备时间
        /// </summary>
        [FieldOffset(4)]
        public xxxdatetime Time;

        /// <summary>
        /// 测温档位
        /// </summary>
        [FieldOffset(4)]
        public int GearLevel;

        /// <summary>
        /// 自动对焦区域
        /// </summary>
        [FieldOffset(4)]
        public xxxfocusrect FocusRect;

        /// <summary>
        /// 位置记忆电动镜头, 配置
        /// </summary>
        [FieldOffset(4)]
        public xf_config XfConfig;

        /// <summary>
        /// 位置记忆电动镜头, 位置-刻度值, [0-1000]
        /// </summary>
        [FieldOffset(4)]
        public int XfMove;

        /// <summary>
        /// 位置记忆电动镜头, 0-计数值, 1-刻度值
        /// </summary>
        [FieldOffset(4)]
        public int XfCurType;

        [FieldOffset(4)]
        public xxxgpiooutput GpioOutput;

        [FieldOffset(4)]
        public short OemHWYZ_Wendu;
        [FieldOffset(6)]
        public short OemHWYZ_Shidu;

        /// <summary>
        /// 外置黑体配置
        /// </summary>
        [FieldOffset(4)]
        public ExtBbConfig BbConfig;


        /// <summary>
        /// 云台指令
        /// </summary>
        [FieldOffset(4)]
        public xxxptzcmd PtzCmd;


        /// <summary>
        /// 鼠标测温
        /// </summary>
        [FieldOffset(4)]
        public xxxmousemea MouseMea;
    }

    /// <summary>
    /// 温度数据帧头
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 128)]
    public struct DataFrameHeader
    {
        /// <summary>
        /// 宽度
        /// </summary>
        public UInt16 Width;

        /// <summary>
        /// 高度
        /// </summary>
        public UInt16 Height;

        /// <summary>
        /// 数据区压缩大小, 内部使用
        /// </summary>
        public UInt32 ComSize;

        /// <summary>
        /// 数据区数据类型, 内部使用
        /// </summary>
        public byte DataType;

        /// <summary>
        /// 数据区压缩类型, 内部使用
        /// </summary>
        public byte ComType;

        /// <summary>
        /// 帧索引
        /// </summary>
        public UInt16 Index;

        /// <summary>
        /// 全帧温度数据是以16位有符号整数数组存储的, 需要根据Slope和Offset转换为浮点数, 浮点值=整数值/Slope+Offset
        /// </summary>
        public UInt16 Slope;

        /// <summary>
        /// 全帧温度数据是以16位有符号整数数组存储的, 需要根据Slope和Offset转换为浮点数, 浮点值=整数值/Slope+Offset
        /// </summary>
        public Int16 Offset;

        /// <summary>
        /// 探测器温度, 单位m°C
        /// </summary>
        public Int32 FPATemp;

        /// <summary>
        /// 等效机壳温度, 单位m°C
        /// </summary>
        public Int32 ShellTemp;

        /// <summary>
        /// 是否已修正
        /// </summary>
        public byte Fixed;

        /// <summary>
        /// GPIO输入0
        /// </summary>
        public byte GpioInput0;

        /// <summary>
        /// GPIO输入1
        /// </summary>
        public byte GpioInput1;

        byte pad1;
        int pad2;

        /// <summary>
        /// 帧时间戳
        /// </summary>
        public Int64 Timestamp;

        //public int c1CaliTemp;
        //public int c1ShellTemp;
        //public int c1ShellTempDelta;
    }

    /// <summary>
    /// 登入信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct YoseenLoginInfo
    {
        /// <summary>
        /// 设备地址
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
        public string CameraAddr;

        /// <summary>
        /// 用户名
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string Username;

        /// <summary>
        /// 密码
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string Password;
    }

    /// <summary>
    /// 登入信息V2, 端口=0, 表示使用默认端口
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct YoseenLoginInfoV2
    {
        /// <summary>
        /// 设备地址
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 128)]
        public string CameraAddr;

        /// <summary>
        /// 用户名
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string Username;

        /// <summary>
        /// 密码
        /// </summary>
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string Password;

        /// <summary>
        /// 配置端口, 默认55502
        /// </summary>
        public ushort PortConfig;

        /// <summary>
        /// 温度流端口, 默认55503
        /// </summary>
        public ushort PortTemp;

        /// <summary>
        /// 温度单帧端口, 默认55505
        /// </summary>
        public ushort PortTempSingle;

        /// <summary>
        /// 视频流端口, 默认55506, 非卡片机使用此端口传输视频数据
        /// </summary>
        public ushort PortVideo;

        /// <summary>
        /// 混合流端口, 默认55507, 卡片机使用此端口传输视频数据
        /// </summary>
        public ushort PortMix;
    }

    /// <summary>
    /// H264起始帧
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 256)]
    public struct H264FrameStart
    {
        /// <summary>
        /// 编码器数据大小
        /// </summary>
        public uint EncDataSize;

        /// <summary>
        /// 数据宽度
        /// </summary>
        public ushort Width;

        /// <summary>
        /// 数据高度
        /// </summary>
        public ushort Height;

        /// <summary>
        /// 编码器数据
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 120)]
        public byte[] EncData;

        /// <summary>
        /// 帧率分子
        /// </summary>
        public ushort FpsNum;

        /// <summary>
        /// 帧率分母
        /// </summary>
        public ushort FpsDen;
    }

    /// <summary>
    /// H264帧测温结果
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct H264_MeasureResult
    {
        /// <summary>
        /// 序号; 0:全局测温结果; 1:中心测温结果; [2-9]:局部测温结果; x无效测温结果
        /// </summary>
        public sbyte Index;

        byte pad1;
        byte pad2;
        byte pad3;

        /// <summary>
        /// 最高温
        /// </summary>
        public float Max;

        /// <summary>
        /// 最低温
        /// </summary>
        public float Min;

        /// <summary>
        /// 平均温
        /// </summary>
        public float Avg;

        /// <summary>
        /// 最高温点坐标X
        /// </summary>
        public ushort MaxX;

        /// <summary>
        /// 最高温点坐标Y
        /// </summary>
        public ushort MaxY;

        /// <summary>
        /// 最低温点坐标X
        /// </summary>
        public ushort MinX;

        /// <summary>
        /// 最低温点坐标Y
        /// </summary>
        public ushort MinY;
    }

    /// <summary>
    /// H264帧头
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Size = 1024)]
    public struct H264FrameHeader
    {
        /// <summary>
        /// 帧大小
        /// </summary>
        public int Size;

        /// <summary>
        /// 数据宽度
        /// </summary>
        public ushort Width;

        /// <summary>
        /// 数据高度
        /// </summary>
        public ushort Height;

        /// <summary>
        /// 探测器温度
        /// </summary>
        public float FPATemp;

        /// <summary>
        /// 全局测温结果
        /// </summary>
        public H264_MeasureResult GlobalResult;

        /// <summary>
        /// 中心测温结果
        /// </summary>
        public H264_MeasureResult CenterResult;

        /// <summary>
        /// 局部测温结果
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public H264_MeasureResult[] LocalResults;

        public static H264FrameHeader Create()
        {
            H264FrameHeader x = new H264FrameHeader();
            x.LocalResults = new H264_MeasureResult[8];
            return x;
        }
    }

    /// <summary>
    /// 温度帧
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct DataFrame
    {
        /// <summary>
        /// 温度数据帧头
        /// </summary>
        public IntPtr Head;

        /// <summary>
        /// 温度数据
        /// </summary>
        public IntPtr Temp;

        /// <summary>
        /// 温度数据压缩
        /// </summary>
        public IntPtr Com;

        /// <summary>
        /// 位图数据
        /// </summary>
        public IntPtr Bmp;

        /// <summary>
        /// H264帧
        /// </summary>
        public IntPtr H264;

        /// <summary>
        /// 前一帧
        /// </summary>
        public IntPtr prev;


        public void Reset()
        {
            Head = IntPtr.Zero;
            Temp = IntPtr.Zero;
            Bmp = IntPtr.Zero;
            H264 = IntPtr.Zero;
            Com = IntPtr.Zero;
            prev = IntPtr.Zero;
        }
    }

    /// <summary>
    /// 预览回调
    /// </summary>
    /// <param name="errorCode">错误码</param>
    /// <param name="dataFrame">数据帧</param>
    /// <param name="customData">用户数据</param>
    public delegate void YoseenPreviewCallback(int errorCode, ref DataFrame dataFrame, IntPtr customData);

    /// <summary>
    /// 预览信息
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct YoseenPreviewInfo
    {
        /// <summary>
        /// xxxdatatype_video 视频流, xxxdatatype_temp 温度流
        /// </summary>
        public int DataType;

        /// <summary>
        /// 显示窗口句柄, 可以为空
        /// </summary>
        public IntPtr Hwnd;

        /// <summary>
        /// 预览回调
        /// </summary>
        public YoseenPreviewCallback CustomCallback;

        /// <summary>
        /// 用户数据
        /// </summary>
        public IntPtr CustomData;

        /// <summary>
        /// 显示画面宽度
        /// </summary>
        public ushort OutputWidth;

        /// <summary>
        /// 显示画面高度
        /// </summary>
        public ushort OutputHeight;
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct YoseenRtspInfo
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 256)]
        public string Url;

        public byte TransMode;

        public byte CacheTime;//0 close, 1-10 cache seconds

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 6)]
        byte[] pad;

        //
        public ushort DataWidth;
        public ushort DataHeight;
    }


    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct ConfigIPReq
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string CameraId;

        public CameraNetworkInfo NetworkInfo;
    }

    /// <summary>
    /// 数据类型
    /// </summary>
    public enum xxxdatatype
    {
        /// <summary>
        /// 温度流
        /// </summary>
        [Description("温度流")]
        temp = 2,

        /// <summary>
        /// 视频流
        /// </summary>
        [Description("视频流")]
        video = 3,

        /// <summary>
        /// 视频流rtsp
        /// </summary>
        [Description("视频流rtsp")]
        video_rtsp = 4,

        /// <summary>
        /// 视频流mix
        /// </summary>
        [Description("视频流mix")]
        video_mix = 5,
    }

    /// <summary>
    /// 数据文件类型
    /// </summary>
    public enum xxxmediafile
    {
        /// <summary>
        /// 单帧温度文件, bmp扩展
        /// </summary>
        bmpx = 0,

        /// <summary>
        /// 单帧温度文件, png扩展
        /// </summary>
        pngx,

        /// <summary>
        /// 多帧温度文件
        /// </summary>
        stream,

        /// <summary>
        /// mp4文件
        /// </summary>
        mp4,

        /// <summary>
        /// 单帧温度文件, jpg扩展
        /// </summary>
        jpgx,
    }

    /// <summary>
    /// 设备文件类型
    /// </summary>
    public enum xxxcamerafile
    {
        /// <summary>
        /// 设备程序包
        /// </summary>
        bin = 0,

        /// <summary>
        /// 校准表
        /// </summary>
        cali,

        /// <summary>
        /// 
        /// </summary>
        com,

        /// <summary>
        /// 日志
        /// </summary>
        log,

        /// <summary>
        /// 校准表2
        /// </summary>
        cali2,

        /// <summary>
        /// 校准表3
        /// </summary>
        cali3,

        /// <summary>
        /// 坏点表
        /// </summary>
        bad,

        /// <summary>
        /// 二次多段校准
        /// </summary>
        c2m,

        /// <summary>
        /// 二次多段校准2
        /// </summary>
        c2m2,

        /// <summary>
        /// 用户校准
        /// </summary>
        cust,

        /// <summary>
        /// 用户校准2
        /// </summary>
        cust2,
    }

    /// <summary>
    /// 调色板类型
    /// </summary>
    public enum xxxpalette : byte
    {
        Autumn = 0,
        BlackHot,
        GlowBow,
        HighContrast,
        HotMetal,
        IronBow,
        Jet,
        RainBow,
        RedSaturation,
        WhiteHot,
        Winter,
        IronBow2,
    }

    /// <summary>
    /// 探测器积分电容
    /// </summary>
    public enum xxxfpacint : byte
    {
        fpacint_0 = 0,
        fpacint_1,
        fpacint_2,
        fpacint_3,
        fpacint_4,
        fpacint_5,
        fpacint_6,
        fpacint_7,
    };

    /// <summary>
    /// 测温结果
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxmeasure_result
    {
        /// <summary>
        /// 最小值
        /// </summary>
        public float amin;

        /// <summary>
        /// 最大值
        /// </summary>
        public float amax;

        /// <summary>
        /// 平均值
        /// </summary>
        public float aavg;

        /// <summary>
        /// 最小值, 整数
        /// </summary>
        public short min;

        /// <summary>
        /// 最大值, 整数
        /// </summary>
        public short max;

        /// <summary>
        /// 平均值, 整数
        /// </summary>
        public short avg;

        /// <summary>
        /// 最大值坐标x
        /// </summary>
        public ushort max_x;

        /// <summary>
        /// 最大值坐标y
        /// </summary>
        public ushort max_y;


        /// <summary>
        /// 最小值坐标x
        /// </summary>
        public ushort min_x;

        /// <summary>
        /// 最小值坐标y
        /// </summary>
        public ushort min_y;
    }

    /// <summary>
    /// 点
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxpoint
    {
        /// <summary>
        /// 坐标x
        /// </summary>
        public ushort x;

        /// <summary>
        /// 坐标y
        /// </summary>
        public ushort y;

        /// <summary>
        /// 半径, 以中心(x,y), 边长(radius*2+1)的方块代表点
        /// </summary>
        public ushort radius;
    }

    /// <summary>
    /// 线
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxline
    {
        /// <summary>
        /// 起点x
        /// </summary>
        public ushort x0;

        /// <summary>
        /// 起点y
        /// </summary>
        public ushort y0;

        /// <summary>
        /// 终点x
        /// </summary>
        public ushort x1;

        /// <summary>
        /// 终点y
        /// </summary>
        public ushort y1;

        /// <summary>
        /// 
        /// </summary>
        public ushort radius;
    }

    /// <summary>
    /// 矩形
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxrectangle
    {
        /// <summary>
        /// x最小
        /// </summary>
        public ushort xmin;

        /// <summary>
        /// y最小
        /// </summary>
        public ushort ymin;

        /// <summary>
        /// x最大
        /// </summary>
        public ushort xmax;

        /// <summary>
        /// y最大
        /// </summary>
        public ushort ymax;
    }

    /// <summary>
    /// 测温对象类型
    /// </summary>
    public enum xxxmeasuretype : byte
    {
        /// <summary>
        /// 点
        /// </summary>
        point = 0,

        /// <summary>
        /// 线
        /// </summary>
        line = 1,

        /// <summary>
        /// 矩形
        /// </summary>
        rectangle = 2,

        /// <summary>
        /// 椭圆
        /// </summary>
        ellipse = 3,

        /// <summary>
        /// 折线
        /// </summary>
        polyline,

        /// <summary>
        /// 多边形
        /// </summary>
        polygon
    };


    /// <summary>
    /// 报警类型
    /// </summary>
    public enum xxxalarmtype : byte
    {
        /// <summary>
        /// 无
        /// </summary>
        none = 0,

        /// <summary>
        /// 高温
        /// </summary>
        max,

        /// <summary>
        /// 低温
        /// </summary>
        min,

        /// <summary>
        /// 区间内
        /// </summary>
        mid,

        /// <summary>
        /// 区间外
        /// </summary>
        exmid,

        /// <summary>
        /// 最高温, 区间内
        /// </summary>
        maxmid,
    };

    [Flags]
    public enum xxxmeasureflags : byte
    {
        track_max = 0x01,
        track_min = 0x02,


        plot_max = 0x08,
        plot_min = 0x10,
        plot_avg = 0x20
    }

    /// <summary>
    /// 测温对象
    /// </summary>
    [StructLayout(LayoutKind.Explicit, Size = 56)]
    public struct xxxmeasure_object
    {
        /// <summary>
        /// ID
        /// </summary>
        [FieldOffset(0)]
        public byte measure_id;

        /// <summary>
        /// 类型
        /// </summary>
        [FieldOffset(16)]
        public xxxmeasuretype measure_type;

        /// <summary>
        /// 报警类型
        /// </summary>
        [FieldOffset(17)]
        public xxxalarmtype alarm_type;

        /// <summary>
        /// 标志
        /// </summary>
        [FieldOffset(18)]
        public xxxmeasureflags measure_flags;

        /// <summary>
        /// 报警最小值
        /// </summary>
        [FieldOffset(20)]
        public float alarm_min;

        /// <summary>
        /// 报警最大值
        /// </summary>
        [FieldOffset(24)]
        public float alarm_max;

        /// <summary>
        /// 发射率
        /// </summary>
        [FieldOffset(28)]
        public float emissivity;

        /// <summary>
        /// 点
        /// </summary>
        [FieldOffset(32)]
        public xxxpoint point;

        /// <summary>
        /// 线
        /// </summary>
        [FieldOffset(32)]
        public xxxline line;

        /// <summary>
        /// 矩形
        /// </summary>
        [FieldOffset(32)]
        public xxxrectangle rectangle;
    }

    /// <summary>
    /// 串口数据位
    /// </summary>
    public enum xxxspdatabits : byte
    {
        databits_5 = 5,
        databits_6,
        databits_7,
        databits_8,
    };

    /// <summary>
    /// 串口校验位
    /// </summary>
    public enum xxxspparity : byte
    {
        none = 0,
        odd,
        even,
    };

    /// <summary>
    /// 串口停止位
    /// </summary>
    public enum xxxspstopbits : byte
    {
        stopbits_1 = 0,
        stopbits_2,
    };

    /// <summary>
    /// 串口波特率
    /// </summary>
    public enum xxxspbaudrate : byte
    {
        baudrate_2400 = 0,
        baudrate_4800,
        baudrate_9600,

        baudrate_19200,
        baudrate_38400,
        baudrate_57600,
        baudrate_115200,
    };

    /// <summary>
    /// 串口用法
    /// </summary>
    public enum xxxspusage : byte
    {
        custom = 0,
        pelcod,
        pelcop
    };

    /// <summary>
    /// GPIO输入类型
    /// </summary>
    public enum xxxgpioinputtype : byte
    {
        /// <summary>
        /// 关闭
        /// </summary>
        none = 0,

        /// <summary>
        /// 检查
        /// </summary>
        check,

        /// <summary>
        /// 截图
        /// </summary>
        capture,
    }

    /// <summary>
    /// GPIO输出类型
    /// </summary>
    public enum xxxgpiooutputtype : byte
    {
        /// <summary>
        /// 关闭
        /// </summary>
        close = 0,

        /// <summary>
        /// 打开
        /// </summary>
        open,
    }

    /// <summary>
    /// 对焦区域
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxfocusrect
    {
        /// <summary>
        /// x最小
        /// </summary>
        public ushort xmin;

        /// <summary>
        /// y最小
        /// </summary>
        public ushort ymin;

        /// <summary>
        /// x最大
        /// </summary>
        public ushort xmax;

        /// <summary>
        /// y最大
        /// </summary>
        public ushort ymax;
    }

    /// <summary>
    /// GPIO报警类型
    /// </summary>
    public enum xxxgpioalarmtype : byte
    {
        /// <summary>
        /// 手动
        /// </summary>
        manual = 0,

        /// <summary>
        /// 自动
        /// </summary>
        auto
    }

    /// <summary>
    /// GPIO报警值
    /// </summary>
    public enum xxxgpioalarmvalue : byte
    {
        /// <summary>
        /// 低
        /// </summary>
        low = 0,

        /// <summary>
        /// 高
        /// </summary>
        high,

        /// <summary>
        /// 无
        /// </summary>
        ignore,
    }

    /// <summary>
    /// GPIO报警
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxgpioalarm
    {
        public xxxgpioalarmvalue value0;		///< 报警值0
        public xxxgpioalarmtype type0;			///< 报警类型0
        public byte time0;			            ///< 报警时间0

        public xxxgpioalarmvalue value1;		///< 报警值1
        public xxxgpioalarmtype type1;			///< 报警类型1
        public byte time1;			            ///< 报警时间1
    }

    /// <summary>
    /// 串口数据
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxserialportdata
    {
        /// <summary>
        /// 数据大小
        /// </summary>
        public byte size;

        /// <summary>
        /// 数据区
        /// </summary>
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 31)]
        public byte[] data;
    }

    /// <summary>
    /// 时间
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    public struct xxxdatetime
    {
        public int year;		///< 年
        public int mon;			///< 月
        public int day;			///< 日

        public int hour;		///< 时
        public int min;			///< 分
        public int sec;			///< 秒
    }

    public enum TransformType : byte
    {
        None = 0,
        Rotate90,
        Rotate180,
        Rotate270,

        //
        MirrorX,
        MirrorY,
    }

    public enum DivisionType : ushort
    {
        Division_1 = 1,
        Division_2 = 2,
        Division_3 = 3,
        Division_4 = 4,
        Division_5 = 5,
        Division_8 = 8,
        Division_16 = 16,
    }

    public enum ZoomType : byte
    {
        Zoom_1 = 1,
        Zoom_2 = 2,
        Zoom_4 = 4
    }


    public enum GearLevel
    {
        Auto,
        Level1,
        Level2,
        Level3,
    }

    [StructLayout(LayoutKind.Sequential, Size = 16)]
    public struct xf_config
    {
        public short ms_full;//ms, 5000
        public short fc_full;//fc, 3200
        public short fc1;//fc
        public short fc2;//fc

        public short fc_delta;//fc
        public short fc_small;//fc, 300
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Size = 128)]
    public struct FPAInfoHK160
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 80)]
        public byte[] data;

        public byte res64;
        public byte cint;       //5
        public ushort tint;     //160
        public ushort vtemp;    //140
        public byte bias;       //0-7v, 1-5v

        public static FPAInfoHK160 create()
        {
            FPAInfoHK160 x = new FPAInfoHK160();
            x.data = new byte[80];
            return x;
        }

        public bool str2data(string str)
        {
            if (str == null) return false;
            char[] dataStr = str.ToCharArray();
            if (dataStr.Length != 80) return false;

            int i, j;
            for (i = 0, j = 0; i < 40; i++, j += 2)
            {
                int d1 = dataStr[j];
                int d0 = dataStr[j + 1];

                if (d1 < 0x41) d1 = d1 - 0x30;
                else d1 = d1 - 0x41 + 10;

                if (d0 < 0x41) d0 = d0 - 0x30;
                else d0 = d0 - 0x41 + 10;

                int d = (d1 << 4) | d0;
                data[39 - i] = (byte)d;
            }

            return true;
        }

        public string data2str()
        {
            char[] charArray = new char[80];

            int i, j;
            for (i = 39, j = 0; i >= 0; i--, j += 2)
            {
                int d = (data[i] & 0xf0) >> 4;
                if (d < 10) d = d + 0x30;
                else d = (d - 10) + 0x41;
                charArray[j] = Convert.ToChar(d);

                d = data[i] & 0x0f;
                if (d < 10) d = d + 0x30;
                else d = (d - 10) + 0x41;
                charArray[j + 1] = Convert.ToChar(d);
            }

            return new string(charArray);
        }

        public void data2arg()
        {
            //cint, 81-83, 10.1
            int x = data[10] & 0x0e;
            cint = (byte)(x >> 1);

            //tint, 290-299, 36.2
            x = (data[37] << 8) | data[36];
            tint = (ushort)((x & 0x0ffc) >> 2);

            //res64, 100-163, 12.4-20.3
            int count = 0;
            int i, j;
            x = data[12] & 0xf0;
            for (j = 4; j <= 7; j++)
            {
                if ((x & (1 << j)) != 0) count++;
            }
            for (i = 0; i <= 6; i++)
            {
                x = data[13 + i];
                for (j = 0; j <= 7; j++)
                {
                    if ((x & (1 << j)) != 0) count++;
                }
            }
            x = data[20] & 0x0f;
            for (j = 0; j <= 3; j++)
            {
                if ((x & (1 << j)) != 0) count++;
            }
            res64 = (byte)count;

            //lvds, 188,174,40,39; 0101
            //p[23] &= 0xef;//23.4
            //p[21] |= 0x40;//21.6
            //p[5] &= 0xfe;//5.0
            //p[4] |= 0x80;//4.7
            //binary, 202, 25.2
            //p[25] &= 0xfb;

            //vtemp, 225-234, 28.1-29.2
            x = (data[29] << 8) | data[28];
            vtemp = (ushort)((x & 0x07fe) >> 1);
        }

        public override string ToString()
        {
            return data2str()+"\n";
        }
    }

    /// <summary>
    /// 混合模式
    /// </summary>
    public enum MixMode
    {
        /// <summary>
        /// 单红外
        /// </summary>
        MixMode_IR = 0,

        /// <summary>
        /// 单可见
        /// </summary>
        MixMode_VIS,

        /// <summary>
        /// 双波段
        /// </summary>
        MixMode_DSX,

        /// <summary>
        /// alpha混合
        /// </summary>
        MixMode_BLEND,

        /// <summary>
        /// 画中画
        /// </summary>
        MixMode_PIP,
    }

    /// <summary>
    /// 翻转方式
    /// </summary>
    public enum MixFlip
    {
        /// <summary>
        /// 无翻转
        /// </summary>
        Flip_NN = 0,

        /// <summary>
        /// X翻转
        /// </summary>
        Flip_XN,

        /// <summary>
        /// Y翻转
        /// </summary>
        Flip_NY,

        /// <summary>
        /// XY翻转
        /// </summary>
        Flip_XY,
    }

    /// <summary>
    /// 双光混合参数
    /// </summary>
    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Size = 48)]
    public struct CameraOutputInfo
    {
        /// <summary>
        /// 混合模式
        /// </summary>
        public byte MixMode;

        /// <summary>
        /// 混合比例, 1-8
        /// </summary>
        public byte MixRatio;

        /// <summary>
        /// 可见光裁剪X
        /// </summary>
        public ushort CropX;

        /// <summary>
        /// 可见光裁剪Y
        /// </summary>
        public ushort CropY;

        /// <summary>
        /// 可见光裁剪宽度
        /// </summary>
        public ushort CropWidth;

        /// <summary>
        /// 可见光裁剪高度
        /// </summary>
        public ushort CropHeight;

        /// <summary>
        /// 红外算法类型
        /// </summary>
        public byte StrechType;

        /// <summary>
        /// 红外DDE等级
        /// </summary>
        public byte DDELevel;

        /// <summary>
        /// 红外增益
        /// </summary>
        public float Gain;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 6)]
        byte[] pad;

        /// <summary>
        /// 红外调色板
        /// </summary>
        public byte PaletteType;

        /// <summary>
        /// 红外对比度
        /// </summary>
        public byte Contrast;

        /// <summary>
        /// 红外亮度
        /// </summary>
        public byte Brightness;

        /// <summary>
        /// 开启模拟视频
        /// </summary>
        public byte EnableHdmi;

        /// <summary>
        /// 流媒体图像组大小
        /// </summary>
        public ushort H264_GopSize;

        /// <summary>
        /// 流媒体码率, 单位: bps
        /// </summary>
        public uint H264_Bitrate;

        /// <summary>
        /// 红外翻转
        /// </summary>
        public byte FlipIR;

        /// <summary>
        /// 可见光翻转
        /// </summary>
        public byte FlipVIS;

        /// <summary>
        /// 可见光视场X, 内部参数
        /// </summary>
        public ushort XStartVIS;

        /// <summary>
        /// 可见光视场X, 内部参数
        /// </summary>
        public ushort YStartVIS;

        /// <summary>
        /// 
        /// </summary>
        public ushort XuiDisplayFlags;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        byte[] reserved;
    }

    /// <summary>
    /// 设备内存数据类型
    /// </summary>
    public enum xxxcameramem
    {
        measure = 0,
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct bad_file_header
    {
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string camera_id;

        public int width;
        public int height;

        public int count;
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi)]
    public struct cust_file_header
    {
        //basic
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 32)]
        public string camera_id;

        public int version;

        public int bte_count;
    }
}
